#!/usr/bin/env python3
#
# Find existing SSH agent socket or start a new ssh-agent process.
#
# If command line args are provided, they are interpreted as secret keys to
# add to the agent.
#
# The shell commands to set SSH_AUTH_SOCK is printed to stdout.
#
# Suggested .bashrc or .zshrc function:
#
#    agent () {
#        start_agent=$HOME/src/py_ssh_agent/start_agent.py
#        keys="$HOME/.ssh/id_ed25519_sk"
#        eval $($start_agent "$@" $keys)
#    }


import argparse
import glob
import os
import pathlib
import re
import subprocess
import sys

DEBUG = False

# Stores the path of the socket from the last time the agent was started.
SOCK = os.path.expanduser('~/.ssh_last_auth_sock')

# Directory containing the auth sockets.  This depends on the OS.
TMP = os.environ.get('TMPDIR') or '/tmp'


def log(*args):
    print(*args, file=sys.stderr)


def debug(*args):
    if DEBUG:
        log(*args)


def is_sock(fn):
    return pathlib.Path(fn).is_socket()


def is_launchd(fn):
    """Return True if this looks like a socket created by MacOS Launchd.  It
    seems it doesn't support ed25519 format keys.
    """
    return '/Listeners' in fn


def is_running(sock_fn):
    try:
        subprocess.check_output(
            ['ssh-add', '-l'],
            encoding='utf-8',
            env=dict(SSH_AUTH_SOCK=sock_fn),
        )
    except subprocess.CalledProcessError:
        return False
    return True


def is_valid_sock(fn):
    return (
        fn
        and os.path.exists(fn)
        and is_sock(fn)
        and not is_launchd(fn)
        and is_running(fn)
    )


def find_sock_path(output):
    m = re.search(r'SSH_AUTH_SOCK=([^;]+);', output)
    if m:
        return m.group(1)
    return ''


def get_existing_sock():
    files = []

    sock_fn = os.environ.get('SSH_AUTH_SOCK')
    if is_valid_sock(sock_fn):
        # If env var is already set and socket exists, use that.
        return sock_fn

    # Use the saved socket path from previous start.  This should be most
    # reliable way.
    try:
        with open(SOCK, 'r') as fp:
            fn = fp.read().strip()
    except OSError:
        pass
    else:
        files.append(fn)

    # Search in tmp folder for sockets.  Verify that a process with that PID
    # is running.
    for fn in glob.glob(os.path.join(TMP, 'ssh-*/agent.*')):
        if not is_sock(fn):
            continue
        debug(f'found existing sock {fn}')
        files.append(fn)

    def get_mtime(fn):
        return os.stat(fn).st_mtime

    files = [fn for fn in files if is_valid_sock(fn)]
    files.sort(key=get_mtime)
    debug('found files', files)
    if files:
        # Return most recently modified file, hopefully agent is still running.
        return files[-1]
    else:
        return None


def main():
    global DEBUG
    parser = argparse.ArgumentParser()
    parser.add_argument(
        '--debug',
        '-d',
        action='store_true',
        default=False,
        help='Enable debugging output',
    )
    parser.add_argument(
        '--gar', action='store_true', help='Enable GCP service account agent.'
    )
    parser.add_argument(
        'key_files', nargs='*', help='SSH private key files to add to agent.'
    )
    args = parser.parse_args()

    sock_fn = get_existing_sock()
    if not sock_fn:
        output = subprocess.check_output(['ssh-agent'], encoding='utf-8')
        sock_fn = find_sock_path(output)
        if not sock_fn:
            log('Cannot found sock path:')
            log(output)
            sys.exit(1)
        log(f'Started new agent: {sock_fn}')
    else:
        log(f'Using existing agent: {sock_fn}')

    if args.key_files:
        os.environ['SSH_AUTH_SOCK'] = sock_fn
        p = subprocess.Popen(
            ['ssh-add', '-l'],
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            encoding='utf-8',
        )
        out, err = p.communicate()
        if 'agent has no identities' in out:
            # Add secret keys using ssh-add
            log('Adding keys...', ' '.join(args.key_files))
            subprocess.run(['ssh-add'] + args.key_files)
        else:
            log('Agent has keys loaded.')

    if args.gar and 'py_ssh_agent_' not in sock_fn:
        # Start the proxy agent (agent.py) to enable GCP service accounts
        py_agent = os.path.expanduser('~/src/py_ssh_agent/agent.py')
        output = subprocess.check_output(
            [py_agent, '--background', '--sock', sock_fn], encoding='utf-8'
        )
        sock_fn = find_sock_path(output)
        if not sock_fn:
            log('Cannot find socket path:')
            log(output)
            sys.exit(1)

    with open(SOCK, 'w') as fp:
        fp.write(sock_fn)

    print(f'SSH_AUTH_SOCK={sock_fn}; export SSH_AUTH_SOCK')


if __name__ == '__main__':
    main()
